/*
 *  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
 *
 *  This file is part of the Arnold emulator source code distribution.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../cpc/headers.h"
#include "../cpc/cpcglob.h"
#include "cheatdb.h"
#include <ctype.h>
#include "../cpc/cpc.h"

#include "../special/win/InterfaceLoadSave.h"

int SelGame;
int NbreGame;
const char *m_pWinapePokeDatabase;
const char **pGamePtrs;
POKEGAMEINFO pgi;


static void	CheatDatabase_Warning(char *pWarningText)
{
	printf("%s",pWarningText);
}

/* define this for checking on database */
#define CPCEMU_FRIENDLY

/* max buffer space for reading in quoted text */
#define CHEAT_DATABASE_MAX_QUOTED_STRING_LENGTH 128
/* length of quoted text */
static unsigned long CheatDatabase_QuotedText_Length;
/* space to store quoted text */
static unsigned char CheatDatabase_QuotedText[CHEAT_DATABASE_MAX_QUOTED_STRING_LENGTH];

static const unsigned char *pDataPtr, *pDataPtrEnd;


/* is a hex digit if it is a digit (0..9), or a letter in range A..F */
#define is_hex_digit(ch) \
	(isdigit(ch) | (isalpha(ch) && ((toupper(ch)>='A') && (toupper(ch)<='F'))))

/* given a char will return the value of the char */
static unsigned long CheatDatabase_GetHexDigit(char ch)
{
	if (isdigit(ch))
	{
		return ch-'0';
	}

	if (isalpha(ch))
	{
		return toupper(ch)-'A' + 10;
	}

	return 0;
}

/*********************************************************************************************/
long NumericChain(char *str)
{

	if ((str[0] == '&') || (str[0] == '#') || (str[0] == '$'))
	{
		return strtoul(str + 1,NULL,16);
	}
	else if (str[0] == '%')
	{
		return strtoul(str + 1,NULL,2);
	}

	return atol(str);
}


/*********************************************************************************************/

const char *WinapePokeDatabase_ReadWord(const char *pPtr, unsigned short *pNumber)
{
	unsigned short nNumber;
	nNumber = (*pPtr) & 0x0ff;
	++pPtr;

	/*
	nNumber = nNumber<<8;
	nNumber = ((*pPtr) & 0x0ff)<<8;//missing part
	*/
	nNumber = (((*pPtr) & 0x0ff) << 8) + nNumber;

	++pPtr;

	*pNumber = nNumber;

	return pPtr;
}

const char *WinapePokeDatabase_ReadCompressedNumber(const char *pPtr, unsigned long *pNumber)
{
	BOOL bFirst = TRUE;
	BOOL bNegative = FALSE;
	int nNumber = 0;
	char nData;
	int nShift = 0;

	do
	{
		nData = *pPtr;
		++pPtr;

		if (bFirst)
		{
			/* if first byte, then bit 6 identifies if number should
			be negated after it is generated */
			if ((nData & (1 << 6)) != 0)
			{
				bNegative = TRUE;
			}
			/* set bits in number */
			//nNumber = nNumber | (nData & ((1<<6)-1));
			nNumber = nNumber | (nData & 63);

			/* indicate we've done first byte */
			bFirst = FALSE;
			/* number of bits to shift number if there are additional bytes */
			//nShift = 5; //Error ?
			nShift = 6;
		}
		else
		{
			/* there were additional bytes, shift number */
			//nNumber = nNumber<<nShift;
			/* combine bits from this byte, 6 bits of byte define number */
			//nNumber = nNumber | (nData & ((1<<7)-1));
			nNumber = nNumber | ((nData & 127) << nShift);
			/* next shift */
			//nShift = 6;//error ?
			nShift = 7;

		}
	}
	/* if bit 7 is set there are additional bytes to read */
	while ((nData & (1 << 7)) != 0);

	/* negate number now? */
	if (bNegative)
	{
		nNumber = -nNumber;
	}
	*pNumber = nNumber;

	return pPtr;
}



BOOL WinapePokeDatabaseInit(const char *pWinapePokeDatabase, long size)
{
	const char *pPtr;

	WinapePokeDatabase_Free();

	m_pWinapePokeDatabase = (const char *)malloc(size);
	memcpy((void *)m_pWinapePokeDatabase,pWinapePokeDatabase,size);

	pPtr = m_pWinapePokeDatabase;

	if (memcmp(pPtr, "WPOK", 4) == 0)
	{
		unsigned long nGames;
		unsigned long i;

		pPtr += 4;

		//number of game
		pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nGames);

		pGamePtrs = (const char **)malloc(sizeof(unsigned char *)*nGames);

		NbreGame = nGames;

		for (i = 0; i < nGames; i++)
		{

			unsigned long nStringLength;
			unsigned short nIdentifierSize;
			unsigned long nPokes;
			unsigned long p;


			pGamePtrs[i] = pPtr;

			//Name of game - string
			pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);

			// add on name length
			pPtr += nStringLength;

			//identifier size
			pPtr = WinapePokeDatabase_ReadWord(pPtr, &nIdentifierSize);
			//add on identifier size (if zero, IDAddr and IDData not present)
			if (nIdentifierSize != 0)
			{
				//Address of Identifier (Word)
				pPtr += sizeof(unsigned short);
				//Identifier Data 
				pPtr += nIdentifierSize;
			}

			//Number of poke
			pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nPokes);

			for (p = 0; p < nPokes; p++)
			{
				unsigned long nEntries;
				unsigned long e;

				// poke description - string
				pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);
				/* add on description length */
				pPtr += nStringLength;

				// poke comment - string
				pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);
				/* add on coment length */
				pPtr += nStringLength;

				/* data type */
				++pPtr;
				/* reversed */
				++pPtr;
				/* ram bank */
				++pPtr;

				//Number of pokes entries
				pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nEntries);

				for (e = 0; e < nEntries; e++)
				{
					unsigned short nBytes;
					unsigned short nAddress;


					/* number of bytes */
					pPtr = WinapePokeDatabase_ReadWord(pPtr, &nBytes);

					/* address */
					pPtr = WinapePokeDatabase_ReadWord(pPtr, &nAddress);

					/* bytes to poke */
					pPtr += (sizeof(unsigned short)*nBytes);
				}


			}
		}
	}
	return TRUE;
}

void WinapePokeDatabase_Free(void)
{
	if (m_pWinapePokeDatabase != NULL)
	{
		free((void *)m_pWinapePokeDatabase);
		m_pWinapePokeDatabase = NULL;
	}
	if (pGamePtrs != NULL)
	{
		free((void *)pGamePtrs);
		pGamePtrs = NULL;
	}
	
	ClearGamePoke();
}

int GetNumberGame(void)
{
	return NbreGame;
}

BOOL GetNameofGame(int i, char *Name)
{
	const char *tmp;
	unsigned long nStringLength;

	if (i > NbreGame) return FALSE;

	tmp = pGamePtrs[i];

	tmp = WinapePokeDatabase_ReadCompressedNumber(tmp, &nStringLength);

	if (nStringLength > 255)
	{
		nStringLength = 255;
	}

	strncpy(Name,tmp,nStringLength);
	Name[nStringLength] = '\0';

	return TRUE;
}

void ClearGamePoke(void)
{
	int i;
	int NbrePoke = pgi.NbrePoke;

	for (i = 0; i < NbrePoke; i++)
	{
		if (pgi.PokeInfoByGame[i].PokeInfo)
		{
			free(pgi.PokeInfoByGame[i].PokeInfo);
			pgi.PokeInfoByGame[i].PokeInfo = NULL;
		}
	}
	if (pgi.PokeInfoByGame)
	{
		free(pgi.PokeInfoByGame);
		pgi.PokeInfoByGame = NULL;
	}

	pgi.NbrePoke = 0;

}

int SearchGameinMemory(void)
{
	const char *pPtr;

	unsigned long nStringLength;
	unsigned short nIdentifierSize;
	int id;

	for (id = 0; id < NbreGame; id++)
	{

		pPtr = pGamePtrs[id];

		//Name of game - string
		pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);
		//memorise
		strncpy(pgi.Name,pPtr,nStringLength);
		pgi.Name[nStringLength] = '\0';

		// add on name length
		pPtr += nStringLength;

		//identifier size
		pPtr = WinapePokeDatabase_ReadWord(pPtr, &nIdentifierSize);
		//add on identifier size (if zero, IDAddr and IDData not present)
		if (nIdentifierSize != 0)
		{
			int i;
			BOOL check = TRUE;
			const MemoryRange *pRange = CPC_GetDefaultMemoryRange();
			unsigned short Address;

			//memorise adress identifier
			Address = (((unsigned char)pPtr[1])<<8) + (unsigned char)pPtr[0];

			//skip Address of Identifier (Word)
			pPtr += sizeof(unsigned short);


			for (i = 0; i < nIdentifierSize; i++)
			{
				//Read memory to check
				int Mask = 0x0ff; //Need to test values
				int v = 0;
				MemoryRange_ReadByte(pRange, Address+i,&v,&Mask);
				if (v != pPtr[i])
				{
					check = FALSE;
				}
			}

			if (check)
			{
				return id;
			}

			//skip Identifier Data 
			pPtr += nIdentifierSize;
		}

	}
	return -1;
}


BOOL CheatSetGame(int id)
{

	const char *pPtr;

	unsigned long nStringLength;
	unsigned short nIdentifierSize;
	unsigned long nPokes;
	unsigned long p;

	if ( (id > NbreGame) || (id < 0) ) return FALSE;
	SelGame = id;

	pPtr = pGamePtrs[id];

	//Name of game - string
	pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);
	//memorise
	strncpy(pgi.Name,pPtr,nStringLength);
	pgi.Name[nStringLength] = '\0';

	// add on name length
	pPtr += nStringLength;

	//identifier size
	pPtr = WinapePokeDatabase_ReadWord(pPtr, &nIdentifierSize);
	//add on identifier size (if zero, IDAddr and IDData not present)
	if (nIdentifierSize != 0)
	{
		//Address of Identifier (Word)
		pPtr += sizeof(unsigned short);
		//Identifier Data 
		pPtr += nIdentifierSize;
	}

	//clear previous Pokes
	ClearGamePoke();

	//Number of poke
	pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nPokes);
	//memorise them
	pgi.NbrePoke = nPokes;

	//chngae array size
	pgi.PokeInfoByGame = (POKEINFOBYGAME*)malloc(nPokes * sizeof(POKEINFOBYGAME));


	for (p = 0; p < nPokes; p++)
	{
		unsigned long nEntries;
		unsigned long e;

		// poke description - string
		pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);
		//memorise it
		strncpy(pgi.PokeInfoByGame[p].Desc, pPtr, nStringLength);
		pgi.PokeInfoByGame[p].Desc[nStringLength] = '\0';
		// add on description length
		pPtr += nStringLength;

		// poke comment - string
		pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nStringLength);
		//memorise it
		strncpy(pgi.PokeInfoByGame[p].Commment, pPtr, nStringLength);
		pgi.PokeInfoByGame[p].Commment[nStringLength] = '\0';
		// add on coment length
		pPtr += nStringLength;

		// data type
		pgi.PokeInfoByGame[p].type = *pPtr;
		++pPtr;

		// reversed
		pgi.PokeInfoByGame[p].reversed = (pPtr[0] == '0') ? TRUE : FALSE;
		++pPtr;

		// ram bank
		pgi.PokeInfoByGame[p].ram_bank = *pPtr;
		++pPtr;

		//Number of pokes entries
		pPtr = WinapePokeDatabase_ReadCompressedNumber(pPtr, &nEntries);
		//memorise it
		pgi.PokeInfoByGame[p].npokes = nEntries;


		//chnage array size
		pgi.PokeInfoByGame[p].PokeInfo = (POKEINFO*)malloc(nEntries * sizeof(POKEINFO));

		for (e = 0; e < nEntries; e++)
		{
			unsigned short nBytes;
			unsigned short nAddress;

			// number of bytes
			pPtr = WinapePokeDatabase_ReadWord(pPtr, &nBytes);
			//memorize
			pgi.PokeInfoByGame[p].PokeInfo[e].NbrBytes = nBytes;

			// address
			pPtr = WinapePokeDatabase_ReadWord(pPtr, &nAddress);
			//memorise
			pgi.PokeInfoByGame[p].PokeInfo[e].address = nAddress;

			// bytes to poke
			//memorize pointer
			pgi.PokeInfoByGame[p].PokeInfo[e].byte = (unsigned short*)pPtr;

			//add on byte lenght
			pPtr += (sizeof(unsigned short)*nBytes);



		}


	}


	return TRUE;
}


BOOL ApplyPoke(char *data,long Address)
{
		unsigned int Value;
		int i = 0;
		const MemoryRange *pRange = CPC_GetDefaultMemoryRange();
		int NbreByte;

		if ((Address < 0) || (!data)) return FALSE;

		NbreByte = strlen(data);

		if (NbreByte > 20) return FALSE;

		while (NbreByte > 0)
		{

			Value = data[i];

			// using Z80_WR_MEM is more work, may trigger breakpoints that kind of thing
			// using memory range applies it directly to appropiate memory range.
			//
			// default memory range is what the cpu sees (so current ram/rom paging for example)
			MemoryRange_WriteByte(pRange, Address, Value);

			//inc adress for next poke
			Address += 1;
			i += 1;

			//one byte done
			NbreByte -= 1;

		}

		return TRUE;
}

BOOL LoadDatabase(char *file)
{
	if (_tcslen(file)!=0)
	{
		unsigned char *pDatabase;
		unsigned long DatabaseLength;

		LoadFile(file, &pDatabase, &DatabaseLength);

		if (pDatabase!=NULL)
		{
			WinapePokeDatabaseInit((const char*)pDatabase, DatabaseLength);
			free(pDatabase);

			return TRUE;
		}

	}

	return FALSE;
}